<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\JobSeeker;
use App\Models\User;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Storage;

class JobSeekerController extends Controller
{
    public function index(Request $request): View
    {
        $query = JobSeeker::query()->with('user');
        if ($search = $request->string('q')->toString()) {
            $query->whereHas('user', function ($q) use ($search) {
                $q->where('email', 'like', "%{$search}%")
                  ->orWhere('first_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%");
            });
        }
        $seekers = $query->latest('id')->paginate(15)->withQueryString();
        return view('admin.job_seekers.index', compact('seekers'));
    }

    public function create(): View
    {
        $users = User::where('user_type', 'jobseeker')->orderBy('email')->pluck('email', 'id');
        return view('admin.job_seekers.create', compact('users'));
    }

    public function store(Request $request): RedirectResponse
    {
        $data = $request->validate([
            'user_id' => ['required','exists:users,id'],
            'experience_years' => ['required', Rule::in(['0-1','2-3','4-5','6+'])],
            'skills' => ['nullable','string'],
            'resume_file' => ['nullable','file','mimes:pdf,doc,docx','max:2048'],
            'current_position' => ['nullable','string','max:100'],
            'desired_salary' => ['nullable','numeric'],
            'preferred_location' => ['nullable','string','max:100'],
            'education_level' => ['required', Rule::in(['high_school','bachelor','master','phd','other'])],
            'languages' => ['nullable','array'],
            'languages.*' => ['string'],
            'certifications' => ['nullable','string'],
        ]);
        if (!empty($data['languages']) && is_array($data['languages'])) {
            $data['languages'] = implode(',', $data['languages']);
        } else {
            $data['languages'] = '';
        }

        if ($request->hasFile('resume_file')) {
            $path = $request->file('resume_file')->store('resumes', 'public');
            $data['resume_file'] = '/storage/' . $path;
        }

        JobSeeker::create($data);
        return redirect()->route('admin.job-seekers.index')->with('success','تم إضافة المتقدم');
    }

    public function show(JobSeeker $job_seeker): View
    {
        $job_seeker->load('user','applications.job');
        return view('admin.job_seekers.show', ['seeker' => $job_seeker]);
    }

    public function edit(JobSeeker $job_seeker): View
    {
        $users = User::where('user_type', 'jobseeker')->orderBy('email')->pluck('email', 'id');
        return view('admin.job_seekers.edit', ['seeker' => $job_seeker, 'users' => $users]);
    }

    public function update(Request $request, JobSeeker $job_seeker): RedirectResponse
    {
        $data = $request->validate([
            'user_id' => ['required','exists:users,id'],
            'experience_years' => ['required', Rule::in(['0-1','2-3','4-5','6+'])],
            'skills' => ['nullable','string'],
            'resume_file' => ['nullable','file','mimes:pdf,doc,docx','max:2048'],
            'current_position' => ['nullable','string','max:100'],
            'desired_salary' => ['nullable','numeric'],
            'preferred_location' => ['nullable','string','max:100'],
            'education_level' => ['required', Rule::in(['high_school','bachelor','master','phd','other'])],
            'languages' => ['nullable','array'],
            'languages.*' => ['string'],
            'certifications' => ['nullable','string'],
        ]);
        if (!empty($data['languages']) && is_array($data['languages'])) {
            $data['languages'] = implode(',', $data['languages']);
        } else {
            $data['languages'] = '';
        }

        if ($request->hasFile('resume_file')) {
            $path = $request->file('resume_file')->store('resumes', 'public');
            $data['resume_file'] = '/storage/' . $path;
        }

        $job_seeker->update($data);
        return redirect()->route('admin.job-seekers.index')->with('success','تم تحديث بيانات المتقدم');
    }

    public function destroy(JobSeeker $job_seeker): RedirectResponse
    {
        $job_seeker->delete();
        return redirect()->route('admin.job-seekers.index')->with('success','تم حذف المتقدم');
    }
}


