<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Company;
use App\Models\Job;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class JobController extends Controller
{
    public function index(Request $request): View
    {
        $query = Job::query()->with(['company','category']);
        if ($search = $request->string('q')->toString()) {
            $query->where('job_title', 'like', "%{$search}%")
                ->orWhere('location', 'like', "%{$search}%");
        }
        $jobs = $query->latest('id')->paginate(12)->withQueryString();
        return view('admin.jobs.index', compact('jobs'));
    }

    public function create(): View
    {
        $companies = Company::orderBy('company_name')->pluck('company_name', 'id');
        $categories = Category::orderBy('name')->pluck('name', 'id');
        return view('admin.jobs.create', compact('companies','categories'));
    }

    public function store(Request $request): RedirectResponse
    {
        $data = $request->validate([
            'company_id' => ['required','exists:companies,id'],
            'job_title' => ['required','string','max:100'],
            'job_description' => ['required','string'],
            'requirements' => ['nullable','string'],
            'salary_min' => ['nullable','numeric'],
            'salary_max' => ['nullable','numeric','gte:salary_min'],
            'job_type' => ['required', Rule::in(['full_time','part_time','contract','internship'])],
            'experience_level' => ['required', Rule::in(['entry','mid','senior','executive'])],
            'location' => ['nullable','string','max:100'],
            'category_id' => ['nullable','exists:categories,id'],
            'skills_required' => ['nullable','string'],
            'is_active' => ['boolean'],
        ]);
        Job::create($data);
        return redirect()->route('admin.jobs.index')->with('success','تم إنشاء الوظيفة بنجاح');
    }

    public function show(Job $job): View
    {
        $job->load(['company','category']);
        return view('admin.jobs.show', compact('job'));
    }

    public function edit(Job $job): View
    {
        $companies = Company::orderBy('company_name')->pluck('company_name', 'id');
        $categories = Category::orderBy('name')->pluck('name', 'id');
        return view('admin.jobs.edit', compact('job','companies','categories'));
    }

    public function update(Request $request, Job $job): RedirectResponse
    {
        $data = $request->validate([
            'company_id' => ['required','exists:companies,id'],
            'job_title' => ['required','string','max:100'],
            'job_description' => ['required','string'],
            'requirements' => ['nullable','string'],
            'salary_min' => ['nullable','numeric'],
            'salary_max' => ['nullable','numeric','gte:salary_min'],
            'job_type' => ['required', Rule::in(['full_time','part_time','contract','internship'])],
            'experience_level' => ['required', Rule::in(['entry','mid','senior','executive'])],
            'location' => ['nullable','string','max:100'],
            'category_id' => ['nullable','exists:categories,id'],
            'skills_required' => ['nullable','string'],
            'is_active' => ['boolean'],
        ]);
        $job->update($data);
        return redirect()->route('admin.jobs.index')->with('success','تم تحديث الوظيفة');
    }

    public function destroy(Job $job): RedirectResponse
    {
        $job->delete();
        return redirect()->route('admin.jobs.index')->with('success','تم حذف الوظيفة');
    }
}


