<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Job;
use App\Models\JobApplication;
use App\Models\JobSeeker;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class JobApplicationController extends Controller
{
    public function index(Request $request): View
    {
        $query = JobApplication::query()->with(['job','jobSeeker.user']);
        if ($status = $request->string('status')->toString()) {
            $query->where('status', $status);
        }
        $applications = $query->latest('id')->paginate(15)->withQueryString();
        return view('admin.applications.index', compact('applications'));
    }

    public function create(): View
    {
        $jobs = Job::orderBy('id','desc')->pluck('job_title', 'id');
        $seekers = JobSeeker::with('user')->get()->pluck('user.email','id');
        return view('admin.applications.create', compact('jobs','seekers'));
    }

    public function store(Request $request): RedirectResponse
    {
        $data = $request->validate([
            'job_id' => ['required','exists:jobs,id'],
            'job_seeker_id' => ['required','exists:job_seekers,id'],
            'cover_letter' => ['nullable','string'],
            'resume_file' => ['nullable','string','max:255'],
            'status' => ['required', Rule::in(['pending','reviewed','shortlisted','interview','accepted','rejected'])],
            'applied_at' => ['nullable','date'],
            'reviewed_at' => ['nullable','date','after_or_equal:applied_at'],
            'notes' => ['nullable','string'],
        ]);
        JobApplication::create($data);
        return redirect()->route('admin.applications.index')->with('success','تم إضافة الطلب');
    }

    public function show(JobApplication $application): View
    {
        $application->load(['job','jobSeeker.user']);
        return view('admin.applications.show', compact('application'));
    }

    public function edit(JobApplication $application): View
    {
        $jobs = Job::orderBy('id','desc')->pluck('job_title', 'id');
        $seekers = JobSeeker::with('user')->get()->pluck('user.email','id');
        return view('admin.applications.edit', compact('application','jobs','seekers'));
    }

    public function update(Request $request, JobApplication $application): RedirectResponse
    {
        $data = $request->validate([
            'job_id' => ['required','exists:jobs,id'],
            'job_seeker_id' => ['required','exists:job_seekers,id'],
            'cover_letter' => ['nullable','string'],
            'resume_file' => ['nullable','string','max:255'],
            'status' => ['required', Rule::in(['pending','reviewed','shortlisted','interview','accepted','rejected'])],
            'applied_at' => ['nullable','date'],
            'reviewed_at' => ['nullable','date','after_or_equal:applied_at'],
            'notes' => ['nullable','string'],
        ]);
        $application->update($data);
        return redirect()->route('admin.applications.index')->with('success','تم تحديث الطلب');
    }

    public function destroy(JobApplication $application): RedirectResponse
    {
        $application->delete();
        return redirect()->route('admin.applications.index')->with('success','تم حذف الطلب');
    }
}


