<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Company;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Contracts\View\View;

class CompanyController extends Controller
{
    public function index(Request $request): View
    {
        $query = Company::query()->with('user');
        if ($search = $request->string('q')->toString()) {
            $query->where('company_name', 'like', "%{$search}%")
                ->orWhere('industry', 'like', "%{$search}%")
                ->orWhere('location', 'like', "%{$search}%");
        }
        $companies = $query->latest('id')->paginate(12)->withQueryString();
        return view('admin.companies.index', compact('companies'));
    }

    public function create(): View
    {
        $users = User::where('user_type', 'company')->pluck('email', 'id');
        return view('admin.companies.create', compact('users'));
    }

    public function store(Request $request): RedirectResponse
    {
        $data = $request->validate([
            'user_id' => ['required','exists:users,id'],
            'company_name' => ['required','string','max:100'],
            'industry' => ['required','string','max:50'],
            'company_email' => ['required','email','max:100'],
            'company_phone' => ['nullable','string','max:20'],
            'company_website' => ['nullable','url','max:255'],
            'company_description' => ['nullable','string'],
            'company_logo' => ['nullable','string','max:255'],
            'company_size' => ['required', Rule::in(['1-10','11-50','51-200','201-500','500+'])],
            'location' => ['nullable','string','max:100'],
            'rating' => ['nullable','numeric','between:0,5'],
            'is_verified' => ['boolean'],
        ]);

        Company::create($data);
        return redirect()->route('admin.companies.index')->with('success', 'تم إنشاء الشركة بنجاح');
    }

    public function show(Company $company): View
    {
        $company->load('user','jobs');
        return view('admin.companies.show', compact('company'));
    }

    public function edit(Company $company): View
    {
        $users = User::where('user_type', 'company')->pluck('email', 'id');
        return view('admin.companies.edit', compact('company','users'));
    }

    public function update(Request $request, Company $company): RedirectResponse
    {
        $data = $request->validate([
            'user_id' => ['required','exists:users,id'],
            'company_name' => ['required','string','max:100'],
            'industry' => ['required','string','max:50'],
            'company_email' => ['required','email','max:100'],
            'company_phone' => ['nullable','string','max:20'],
            'company_website' => ['nullable','url','max:255'],
            'company_description' => ['nullable','string'],
            'company_logo' => ['nullable','string','max:255'],
            'company_size' => ['required', Rule::in(['1-10','11-50','51-200','201-500','500+'])],
            'location' => ['nullable','string','max:100'],
            'rating' => ['nullable','numeric','between:0,5'],
            'is_verified' => ['boolean'],
        ]);

        $company->update($data);
        return redirect()->route('admin.companies.index')->with('success', 'تم تحديث الشركة');
    }

    public function destroy(Company $company): RedirectResponse
    {
        $company->delete();
        return redirect()->route('admin.companies.index')->with('success', 'تم حذف الشركة');
    }
}


